/* drivers/input/misc/gme60X_compass.c - gme60X/gmc30x compass driver
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#define DEBUG
//#define VERBOSE_DEBUG

#include <linux/gme60x_compass.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/freezer.h>
#include <linux/gpio.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/workqueue.h>

#define GME_DEBUG_IF			0
#define GME_HAS_RESET			1
#define GME_INPUT_DEVICE_NAME	"compass"
#define GME_DRDY_TIMEOUT_MS		100
#define GME_BASE_NUM			10
#define QCOM        1
#define GME60x_DEBUG_MSG 1
#if GME60x_DEBUG_MSG
#define GMEDBG(format, ...)	printk(KERN_ERR "GME60x " format "\n", ## __VA_ARGS__)
#else
#define GMEDBG(format, ...)
#endif

struct gme_compass_data {
	struct i2c_client	*i2c;
	struct input_dev	*input;
	struct device		*class_dev;
	struct class		*compass;

	wait_queue_head_t	drdy_wq;
	wait_queue_head_t	open_wq;

	/* These two buffers are initialized at start up.
	   After that, the value is not changed */
	uint8_t sense_info[GME_SENSOR_INFO_SIZE];
	uint8_t sense_conf[GME_SENSOR_CONF_SIZE];

	struct	mutex sensor_mutex;
	uint8_t	sense_data[GME_SENSOR_DATA_SIZE];
	struct mutex accel_mutex;
	int16_t accel_data[3];

	/* Positive value means the device is working.
	   0 or negative value means the device is not woking,
	   i.e. in power-down mode. */
	int8_t	is_busy;

	struct mutex	val_mutex;
	uint32_t		enable_flag;
	int64_t			delay[GME_NUM_SENSORS];

	atomic_t	active;
	atomic_t	drdy;

	char layout;
	int	irq;
	int	gpio_rstn;
};

static struct gme_compass_data *s_gme;

int sGx=0;

/***** I2C I/O function ***********************************************/
static int gme_rxdata(
	struct i2c_client *i2c,
	uint8_t *rxData,
	int length)
{
	int ret;

	struct i2c_msg msgs[] = {
		{
			.addr = i2c->addr,
			.flags = 0,
			.len = 1,
			.buf = rxData,
		},
		{
			.addr = i2c->addr,
			.flags = I2C_M_RD,
			.len = length,
			.buf = rxData,
		},
	};
	uint8_t addr = rxData[0];

	ret = i2c_transfer(i2c->adapter, msgs, ARRAY_SIZE(msgs));
	if (ret < 0) {
		dev_err(&i2c->dev, "%s: transfer failed.", __func__);
		return ret;
	} else if (ret != ARRAY_SIZE(msgs)) {
		dev_err(&i2c->dev, "%s: transfer failed(size error).\n",
				__func__);
		return -ENXIO;
	}

	dev_vdbg(&i2c->dev, "RxData: len=%02x, addr=%02x, data=%02x",
		length, addr, rxData[0]);

	return 0;
}

static int gme_txdata(
	struct i2c_client *i2c,
	uint8_t *txData,
	int length)
{
	int ret;

	struct i2c_msg msg[] = {
		{
			.addr = i2c->addr,
			.flags = 0,
			.len = length,
			.buf = txData,
		},
	};

	ret = i2c_transfer(i2c->adapter, msg, ARRAY_SIZE(msg));
	if (ret < 0) {
		dev_err(&i2c->dev, "%s: transfer failed.", __func__);
		return ret;
	} else if (ret != ARRAY_SIZE(msg)) {
		dev_err(&i2c->dev, "%s: transfer failed(size error).",
				__func__);
		return -ENXIO;
	}

	dev_vdbg(&i2c->dev, "TxData: len=%02x, addr=%02x data=%02x",
		length, txData[0], txData[1]);

	return 0;
}

/***** compass miscdevice functions *************************************/
static int AKECS_Set_CNTL(
	struct gme_compass_data *data,
	uint8_t mode)
{
	uint8_t buffer[2];
	int err;

	/***** lock *****/
	mutex_lock(&data->sensor_mutex);
	/* Busy check */
	if (data->is_busy > 0) {
		dev_err(&data->i2c->dev,
				"%s: device is busy.", __func__);
		err = -EBUSY;
	} else {
		/* Set measure mode */
		buffer[0] = GME_REG_MODE;
		buffer[1] = mode;
		err = gme_txdata(data->i2c, buffer, 2);
		if (err < 0) {
			dev_err(&data->i2c->dev,
					"%s: Can not set CNTL.", __func__);
		} else {
			dev_vdbg(&data->i2c->dev,
					"Mode is set to (%d).", mode);
			/* Set flag */
			data->is_busy = 1;
			atomic_set(&data->drdy, 0);
			/* wait at least 100us after changing mode */
			udelay(100);
		}
	}

	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

	return err;
}

static int AKECS_Set_PowerDown(
	struct gme_compass_data *data)
{
	uint8_t buffer[2];
	int err;

	/***** lock *****/
	mutex_lock(&data->sensor_mutex);

	/* Set powerdown mode */
	buffer[0] = GME_REG_MODE;
	buffer[1] = GME_MODE_POWERDOWN;
	err = gme_txdata(data->i2c, buffer, 2);
	if (err < 0) {
		dev_err(&data->i2c->dev,
			"%s: Can not set to powerdown mode.", __func__);
	} else {
		dev_dbg(&data->i2c->dev, "Powerdown mode is set.");
		/* wait at least 100us after changing mode */
		udelay(100);
	}
	/* Clear status */
	data->is_busy = 0;
	atomic_set(&data->drdy, 0);

	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

	return err;
}

static int AKECS_Reset(
	struct gme_compass_data *data,
	int hard)
{
	int err;

#if GME_HAS_RESET
	uint8_t buffer[2];

	/***** lock *****/
	mutex_lock(&data->sensor_mutex);

	if (hard != 0) {
		gpio_set_value(data->gpio_rstn, 0);
		udelay(5);
		gpio_set_value(data->gpio_rstn, 1);
		/* No error is returned */
		err = 0;
	} else {
		buffer[0] = GME_REG_RESET;
		buffer[1] = GME_RESET_DATA;
		err = gme_txdata(data->i2c, buffer, 2);
		if (err < 0) {
			dev_err(&data->i2c->dev,
				"%s: Can not set SRST bit.", __func__);
		} else {
			dev_dbg(&data->i2c->dev, "Soft reset is done.");
		}
	}
	/* Device will be accessible 100 us after */
	udelay(100);
	/* Clear status */
	data->is_busy = 0;
	atomic_set(&data->drdy, 0);

	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

#else
	err = AKECS_Set_PowerDown(data);
#endif

	return err;
}

static int AKECS_SetMode(
	struct gme_compass_data *data,
	uint8_t mode)
{
	int err;

	switch (mode & 0x1F) {
	case GME_MODE_SNG_MEASURE:
	case GME_MODE_SELF_TEST:
	case GME_MODE_FUSE_ACCESS:
		err = AKECS_Set_CNTL(data, mode);
		break;
	case GME_MODE_POWERDOWN:
		err = AKECS_Set_PowerDown(data);
		break;
	default:
		dev_err(&data->i2c->dev,
			"%s: Unknown mode(%d).", __func__, mode);
		return -EINVAL;
	}

	return err;
}

static void AKECS_SetYPR(
	struct gme_compass_data *data,
	int *rbuf)
{
	uint32_t ready;
	dev_vdbg(&data->i2c->dev, "%s: flag =0x%X", __func__, rbuf[0]);
	dev_vdbg(&data->input->dev, "  Acc [LSB]   : %6d,%6d,%6d stat=%d",
		rbuf[1], rbuf[2], rbuf[3], rbuf[4]);
	dev_vdbg(&data->input->dev, "  Geo [LSB]   : %6d,%6d,%6d stat=%d",
		rbuf[5], rbuf[6], rbuf[7], rbuf[8]);
	dev_vdbg(&data->input->dev, "  Orientation : %6d,%6d,%6d",
		rbuf[9], rbuf[10], rbuf[11]);
	dev_vdbg(&data->input->dev, "  Rotation V  : %6d,%6d,%6d,%6d",
		rbuf[12], rbuf[13], rbuf[14], rbuf[15]);

	/* No events are reported */
	if (!rbuf[0]) {
		dev_dbg(&data->i2c->dev, "Don't waste a time. No events are reported !");
		return;
	}

	mutex_lock(&data->val_mutex);
	ready = (data->enable_flag & (uint32_t)rbuf[0]);
	mutex_unlock(&data->val_mutex);

	/* Report acceleration sensor information */
	if (ready & ACC_DATA_READY) {
		input_report_abs(data->input, ABS_X, rbuf[1]);
		input_report_abs(data->input, ABS_Y, rbuf[2]);
		input_report_abs(data->input, ABS_Z, rbuf[3]);
		input_report_abs(data->input, ABS_RX, rbuf[4]);
	}
	/* Report magnetic vector information */
	if (ready & MAG_DATA_READY) {
		input_report_abs(data->input, ABS_RY, rbuf[5]);
		input_report_abs(data->input, ABS_RZ, rbuf[6]);
		input_report_abs(data->input, ABS_THROTTLE, rbuf[7]);
		input_report_abs(data->input, ABS_RUDDER, rbuf[8]);
	}
	/* Report fusion sensor information */
	if (ready & FUSION_DATA_READY) {
		/* Orientation */
		input_report_abs(data->input, ABS_HAT0Y, rbuf[9]);
		input_report_abs(data->input, ABS_HAT1X, rbuf[10]);
		input_report_abs(data->input, ABS_HAT1Y, rbuf[11]);
		/* Rotation Vector */
		input_report_abs(data->input, ABS_TILT_X, rbuf[12]);
		input_report_abs(data->input, ABS_TILT_Y, rbuf[13]);
		input_report_abs(data->input, ABS_TOOL_WIDTH, rbuf[14]);
		input_report_abs(data->input, ABS_VOLUME, rbuf[15]);
		/* Soft Gyro */
		if(sGx>=5000)
			sGx=-5000;
		input_report_abs(data->input, ABS_HAT2X, rbuf[16]);
		input_report_abs(data->input, ABS_HAT2Y, rbuf[17]);
		input_report_abs(data->input, ABS_HAT3X, rbuf[18]);
	}

	input_sync(data->input);
}

/* This function will block a process until the latest measurement
 * data is available.
 */
static int AKECS_GetData(
	struct gme_compass_data *data,
	uint8_t *rbuf,
	int size)
{
	int err;

	/* Block! */
	err = wait_event_interruptible_timeout(
			data->drdy_wq,
			atomic_read(&data->drdy),
			msecs_to_jiffies(GME_DRDY_TIMEOUT_MS));

	if (err < 0) {
		dev_err(&data->i2c->dev,
			"%s: wait_event failed (%d).", __func__, err);
		return err;
	}
	if (!atomic_read(&data->drdy)) {
		dev_err(&data->i2c->dev,
			"%s: DRDY is not set.", __func__);
		return -ENODATA;
	}

	/***** lock *****/
	mutex_lock(&data->sensor_mutex);

	memcpy(rbuf, data->sense_data, size);
	atomic_set(&data->drdy, 0);

	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

	return 0;
}

static int AKECS_GetData_Poll(
	struct gme_compass_data *data,
	uint8_t *rbuf,
	int size)
{
	uint8_t buffer[GME_SENSOR_DATA_SIZE];
	int err;
	/* Read status */
	buffer[0] = GME_REG_STATUS;
	err = gme_rxdata(data->i2c, buffer, 1);
	if (err < 0) {
		dev_err(&data->i2c->dev, "%s failed.", __func__);
		return err;
	}
	//GMEDBG("%s,%d,DRDY= %x\n",__func__,__LINE__,buffer[0]);

	/* Check ST bit */
	if (!(GME_DRDY_IS_HIGH(buffer[0])))
		return -EAGAIN;
	//GMEDBG("%s,%d,DRDY= %x\n",__func__,__LINE__,buffer[0]);

	/* Read rest data */
	buffer[1] = GME_REG_STATUS + 1;
	err = gme_rxdata(data->i2c, &(buffer[1]), GME_SENSOR_DATA_SIZE-1);
	if (err < 0) {
		dev_err(&data->i2c->dev, "%s failed.", __func__);
		return err;
	}
	memcpy(rbuf, buffer, size);
	atomic_set(&data->drdy, 0);

	/***** lock *****/
	mutex_lock(&data->sensor_mutex);
	data->is_busy = 0;
	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

	return 0;
}

static int AKECS_GetOpenStatus(
	struct gme_compass_data *data)
{
	return wait_event_interruptible(
			data->open_wq, (atomic_read(&data->active) > 0));
}

static int AKECS_GetCloseStatus(
	struct gme_compass_data *data)
{
	return wait_event_interruptible(
			data->open_wq, (atomic_read(&data->active) <= 0));
}

static int AKECS_Open(struct inode *inode, struct file *file)
{
	file->private_data = s_gme;
	return nonseekable_open(inode, file);
}

static int AKECS_Release(struct inode *inode, struct file *file)
{
	return 0;
}

static long
AKECS_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	struct gme_compass_data *data = file->private_data;

	/* NOTE: In this function the size of "char" should be 1-byte. */
	uint8_t i2c_buf[GME_RWBUF_SIZE];		/* for READ/WRITE */
	uint8_t dat_buf[GME_SENSOR_DATA_SIZE];/* for GET_DATA */
	int32_t ypr_buf[GME_YPR_DATA_SIZE];		/* for SET_YPR */
	int64_t delay[GME_NUM_SENSORS];	/* for GET_DELAY */
	int16_t acc_buf[3];	/* for GET_ACCEL */
	uint8_t mode;			/* for SET_MODE*/
	int status;			/* for OPEN/CLOSE_STATUS */
	int ret = 0;		/* Return value. */

	switch (cmd) {
	case ECS_IOCTL_READ:
	case ECS_IOCTL_WRITE:
		if (argp == NULL) {
			dev_err(&data->i2c->dev, "invalid argument.");
			return -EINVAL;
		}
		if (copy_from_user(&i2c_buf, argp, sizeof(i2c_buf))) {
			dev_err(&data->i2c->dev, "copy_from_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_SET_MODE:
		if (argp == NULL) {
			dev_err(&data->i2c->dev, "invalid argument.");
			return -EINVAL;
		}
		if (copy_from_user(&mode, argp, sizeof(mode))) {
			dev_err(&data->i2c->dev, "copy_from_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_SET_YPR:
		if (argp == NULL) {
			dev_err(&data->i2c->dev, "invalid argument.");
			return -EINVAL;
		}
		if (copy_from_user(&ypr_buf, argp, sizeof(ypr_buf))) {
			dev_err(&data->i2c->dev, "copy_from_user failed.");
			return -EFAULT;
		}
	case ECS_IOCTL_GET_INFO:
	case ECS_IOCTL_GET_CONF:
	case ECS_IOCTL_GET_DATA:
	case ECS_IOCTL_GET_OPEN_STATUS:
	case ECS_IOCTL_GET_CLOSE_STATUS:
	case ECS_IOCTL_GET_DELAY:
	case ECS_IOCTL_GET_LAYOUT:
	case ECS_IOCTL_GET_ACCEL:
		/* Check buffer pointer for writing a data later. */
		if (argp == NULL) {
			dev_err(&data->i2c->dev, "invalid argument.");
			return -EINVAL;
		}
		break;
	default:
		break;
	}

	switch (cmd) {
	case ECS_IOCTL_READ:
		dev_vdbg(&data->i2c->dev, "IOCTL_READ called.");
		if ((i2c_buf[0] < 1) || (i2c_buf[0] > (GME_RWBUF_SIZE-1))) {
			dev_err(&data->i2c->dev, "invalid argument.");
			return -EINVAL;
		}
		ret = gme_rxdata(data->i2c, &i2c_buf[1], i2c_buf[0]);
		if (ret < 0)
			return ret;
		break;
	case ECS_IOCTL_WRITE:
		dev_vdbg(&data->i2c->dev, "IOCTL_WRITE called.");
		if ((i2c_buf[0] < 2) || (i2c_buf[0] > (GME_RWBUF_SIZE-1))) {
			dev_err(&data->i2c->dev, "invalid argument.");
			return -EINVAL;
		}
		ret = gme_txdata(data->i2c, &i2c_buf[1], i2c_buf[0]);
		if (ret < 0)
			return ret;
		break;
	case ECS_IOCTL_RESET:
		dev_vdbg(&data->i2c->dev, "IOCTL_RESET called.");
		ret = AKECS_Reset(data, data->gpio_rstn);
		if (ret < 0)
			return ret;
		break;
	case ECS_IOCTL_SET_MODE:
		dev_vdbg(&data->i2c->dev, "IOCTL_SET_MODE called.");
		ret = AKECS_SetMode(data, mode);
		if (ret < 0)
			return ret;
		break;
	case ECS_IOCTL_SET_YPR:
		dev_vdbg(&data->i2c->dev, "IOCTL_SET_YPR called.");
		AKECS_SetYPR(data, ypr_buf);
		break;
	case ECS_IOCTL_GET_DATA:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_DATA called.");
		if (data->irq)
			ret = AKECS_GetData(data, dat_buf, GME_SENSOR_DATA_SIZE);
		else
			ret = AKECS_GetData_Poll(data, dat_buf, GME_SENSOR_DATA_SIZE);

		if (ret < 0)
			return ret;
		break;
	case ECS_IOCTL_GET_OPEN_STATUS:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_OPEN_STATUS called.");
		ret = AKECS_GetOpenStatus(data);
		if (ret < 0) {
			dev_err(&data->i2c->dev,
				"Get Open returns error (%d).", ret);
			return ret;
		}
		break;
	case ECS_IOCTL_GET_CLOSE_STATUS:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_CLOSE_STATUS called.");
		ret = AKECS_GetCloseStatus(data);
		if (ret < 0) {
			dev_err(&data->i2c->dev,
				"Get Close returns error (%d).", ret);
			return ret;
		}
		break;
	case ECS_IOCTL_GET_DELAY:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_DELAY called.");
		mutex_lock(&data->val_mutex);
		delay[0] = ((data->enable_flag & ACC_DATA_READY) ?
				data->delay[0] : -1);
		delay[1] = ((data->enable_flag & MAG_DATA_READY) ?
				data->delay[1] : -1);
		delay[2] = ((data->enable_flag & FUSION_DATA_READY) ?
				data->delay[2] : -1);
		mutex_unlock(&data->val_mutex);
		break;
	case ECS_IOCTL_GET_INFO:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_INFO called.");
		break;
	case ECS_IOCTL_GET_CONF:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_CONF called.");
		break;
	case ECS_IOCTL_GET_LAYOUT:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_LAYOUT called.");
		break;
	case ECS_IOCTL_GET_ACCEL:
		dev_vdbg(&data->i2c->dev, "IOCTL_GET_ACCEL called.");
		mutex_lock(&data->accel_mutex);
		acc_buf[0] = data->accel_data[0];
		acc_buf[1] = data->accel_data[1];
		acc_buf[2] = data->accel_data[2];
		mutex_unlock(&data->accel_mutex);
		break;
	default:
		return -ENOTTY;
	}

	switch (cmd) {
	case ECS_IOCTL_READ:
		/* +1  is for the first byte */
		if (copy_to_user(argp, &i2c_buf, i2c_buf[0]+1)) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_INFO:
		if (copy_to_user(argp, &data->sense_info,
					sizeof(data->sense_info))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_CONF:
		if (copy_to_user(argp, &data->sense_conf,
					sizeof(data->sense_conf))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_DATA:
		if (copy_to_user(argp, &dat_buf, sizeof(dat_buf))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_OPEN_STATUS:
	case ECS_IOCTL_GET_CLOSE_STATUS:
		status = atomic_read(&data->active);
		if (copy_to_user(argp, &status, sizeof(status))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_DELAY:
		if (copy_to_user(argp, &delay, sizeof(delay))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_LAYOUT:
		if (copy_to_user(argp, &data->layout, sizeof(data->layout))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	case ECS_IOCTL_GET_ACCEL:
		if (copy_to_user(argp, &acc_buf, sizeof(acc_buf))) {
			dev_err(&data->i2c->dev, "copy_to_user failed.");
			return -EFAULT;
		}
		break;
	default:
		break;
	}

	return 0;
}

static const struct file_operations AKECS_fops = {
	.owner = THIS_MODULE,
	.open = AKECS_Open,
	.release = AKECS_Release,
	.unlocked_ioctl = AKECS_ioctl,
};

static struct miscdevice gme_compass_dev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = GME_MISCDEV_NAME,
	.fops = &AKECS_fops,
};

/***** gme sysfs functions ******************************************/
static int create_device_attributes(
	struct device *dev,
	struct device_attribute *attrs)
{
	int i;
	int err = 0;

	for (i = 0 ; NULL != attrs[i].attr.name ; ++i) {
		err = device_create_file(dev, &attrs[i]);
		if (err)
			break;
	}

	if (err) {
		for (--i; i >= 0 ; --i)
			device_remove_file(dev, &attrs[i]);
	}

	return err;
}

static void remove_device_attributes(
	struct device *dev,
	struct device_attribute *attrs)
{
	int i;

	for (i = 0 ; NULL != attrs[i].attr.name ; ++i)
		device_remove_file(dev, &attrs[i]);
}

static int create_device_binary_attributes(
	struct kobject *kobj,
	struct bin_attribute *attrs)
{
	int i;
	int err = 0;

	err = 0;

	for (i = 0 ; NULL != attrs[i].attr.name ; ++i) {
		err = sysfs_create_bin_file(kobj, &attrs[i]);
		if (0 != err)
			break;
	}

	if (0 != err) {
		for (--i; i >= 0 ; --i)
			sysfs_remove_bin_file(kobj, &attrs[i]);
	}

	return err;
}

static void remove_device_binary_attributes(
	struct kobject *kobj,
	struct bin_attribute *attrs)
{
	int i;

	for (i = 0 ; NULL != attrs[i].attr.name ; ++i)
		sysfs_remove_bin_file(kobj, &attrs[i]);
}

/*********************************************************************
 *
 * SysFS attribute functions
 *
 * directory : /sys/class/compass/gme60x/
 * files :
 *  - enable_acc    [rw] [t] : enable flag for accelerometer
 *  - enable_mag    [rw] [t] : enable flag for magnetometer
 *  - enable_fusion [rw] [t] : enable flag for fusion sensor
 *  - delay_acc     [rw] [t] : delay in nanosecond for accelerometer
 *  - delay_mag     [rw] [t] : delay in nanosecond for magnetometer
 *  - delay_fusion  [rw] [t] : delay in nanosecond for fusion sensor
 *
 * debug :
 *  - mode       [w]  [t] : E-Compass mode
 *  - bdata      [r]  [t] : buffered raw data
 *  - asa        [r]  [t] : FUSEROM data
 *  - regs       [r]  [t] : read all registers
 *
 * [b] = binary format
 * [t] = text format
 */

/***** sysfs enable *************************************************/
static void gme_compass_sysfs_update_status(
	struct gme_compass_data *data)
{
	uint32_t en;
	mutex_lock(&data->val_mutex);
	en = data->enable_flag;
	mutex_unlock(&data->val_mutex);

	if (en == 0) {
		if (atomic_cmpxchg(&data->active, 1, 0) == 1) {
			wake_up(&data->open_wq);
			dev_dbg(data->class_dev, "Deactivated");
		}
	} else {
		if (atomic_cmpxchg(&data->active, 0, 1) == 0) {
			wake_up(&data->open_wq);
			dev_dbg(data->class_dev, "Activated");
		}
	}
	dev_dbg(&data->i2c->dev,
		"Status updated: enable=0x%X, active=%d",
		en, atomic_read(&data->active));
}

static ssize_t gme_compass_sysfs_enable_show(
	struct gme_compass_data *data, char *buf, int pos)
{
	int flag;

	mutex_lock(&data->val_mutex);
	flag = ((data->enable_flag >> pos) & 1);
	mutex_unlock(&data->val_mutex);

	return scnprintf(buf, PAGE_SIZE, "%d\n", flag);
}

static ssize_t gme_compass_sysfs_enable_store(
	struct gme_compass_data *data, char const *buf, size_t count, int pos)
{
	long en = 0;

	if (NULL == buf)
		return -EINVAL;

	if (0 == count)
		return 0;

	if (strict_strtol(buf, GME_BASE_NUM, &en))
		return -EINVAL;

	en = en ? 1 : 0;

	mutex_lock(&data->val_mutex);
	data->enable_flag &= ~(1<<pos);
	data->enable_flag |= ((uint32_t)(en))<<pos;
	mutex_unlock(&data->val_mutex);

	gme_compass_sysfs_update_status(data);

	return count;
}

/***** Acceleration ***/
static ssize_t gme_enable_acc_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	return gme_compass_sysfs_enable_show(
		dev_get_drvdata(dev), buf, ACC_DATA_FLAG);
}
static ssize_t gme_enable_acc_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	return gme_compass_sysfs_enable_store(
		dev_get_drvdata(dev), buf, count, ACC_DATA_FLAG);
}

/***** Magnetic field ***/
static ssize_t gme_enable_mag_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	return gme_compass_sysfs_enable_show(
		dev_get_drvdata(dev), buf, MAG_DATA_FLAG);
}
static ssize_t gme_enable_mag_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	return gme_compass_sysfs_enable_store(
		dev_get_drvdata(dev), buf, count, MAG_DATA_FLAG);
}

/***** Fusion ***/
static ssize_t gme_enable_fusion_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	return gme_compass_sysfs_enable_show(
		dev_get_drvdata(dev), buf, FUSION_DATA_FLAG);
}
static ssize_t gme_enable_fusion_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	return gme_compass_sysfs_enable_store(
		dev_get_drvdata(dev), buf, count, FUSION_DATA_FLAG);
}

/***** sysfs delay **************************************************/
static ssize_t gme_compass_sysfs_delay_show(
	struct gme_compass_data *data, char *buf, int pos)
{
	int64_t val;

	mutex_lock(&data->val_mutex);
	val = data->delay[pos];
	mutex_unlock(&data->val_mutex);

	return scnprintf(buf, PAGE_SIZE, "%lld\n", val);
}

static ssize_t gme_compass_sysfs_delay_store(
	struct gme_compass_data *data, char const *buf, size_t count, int pos)
{
	long long val = 0;
	
	if (NULL == buf)
		return -EINVAL;

	if (0 == count)
		return 0;

	if (strict_strtoll(buf, GME_BASE_NUM, &val))
		return -EINVAL;

	mutex_lock(&data->val_mutex);
	data->delay[pos] = val;
	mutex_unlock(&data->val_mutex);

	return count;
}

/***** Accelerometer ***/
static ssize_t gme_delay_acc_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	return gme_compass_sysfs_delay_show(
		dev_get_drvdata(dev), buf, ACC_DATA_FLAG);
}
static ssize_t gme_delay_acc_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	return gme_compass_sysfs_delay_store(
		dev_get_drvdata(dev), buf, count, ACC_DATA_FLAG);
}

/***** Magnetic field ***/
static ssize_t gme_delay_mag_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	return gme_compass_sysfs_delay_show(
		dev_get_drvdata(dev), buf, MAG_DATA_FLAG);
}
static ssize_t gme_delay_mag_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	return gme_compass_sysfs_delay_store(
		dev_get_drvdata(dev), buf, count, MAG_DATA_FLAG);
}
/*----------------------------shipment test------------------------------------------------*/
/*!
 @return If @a testdata is in the range of between @a lolimit and @a hilimit,
 the return value is 1, otherwise -1.
 @param[in] testno   A pointer to a text string.
 @param[in] testname A pointer to a text string.
 @param[in] testdata A data to be tested.
 @param[in] lolimit  The maximum allowable value of @a testdata.
 @param[in] hilimit  The minimum allowable value of @a testdata.
 @param[in,out] pf_total
 */
int
TEST_DATA(const char testno[],
		  const char testname[],
          const int testdata,
		  const int lolimit,
		  const int hilimit,
          int * pf_total)
{
	int pf;                     //Pass;1, Fail;-1

	if ((testno == NULL) && (strncmp(testname, "START", 5) == 0)) {
		// Display header
		GMEDBG("--------------------------------------------------------------------\n");
		GMEDBG(" Test No. Test Name    Fail    Test Data    [      Low         High]\n");
		GMEDBG("--------------------------------------------------------------------\n");

		pf = 1;
	} else if ((testno == NULL) && (strncmp(testname, "END", 3) == 0)) {
		// Display result
		GMEDBG("--------------------------------------------------------------------\n");
		if (*pf_total == 1) {
			GMEDBG("Factory shipment test was passed.\n\n");
		} else {
			GMEDBG("Factory shipment test was failed.\n\n");
		}

		pf = 1;
	} else {
		if ((lolimit <= testdata) && (testdata <= hilimit)) {
			//Pass
			pf = 1;
		} else {
			//Fail
			pf = -1;
		}

		//display result
		GMEDBG(" %7s  %-10s      %c    %9d    [%9d    %9d]\n",
				 testno, testname, ((pf == 1) ? ('.') : ('F')), testdata,
				 lolimit, hilimit);
	}

	//Pass/Fail check
	if (*pf_total != 0) {
		if ((*pf_total == 1) && (pf == 1)) {
			*pf_total = 1;            //Pass
		} else {
			*pf_total = -1;           //Fail
		}
	}
	return pf;
}

/*!
 Execute "Onboard Function Test" (NOT includes "START" and "END" command).
 @retval 1 The test is passed successfully.
 @retval -1 The test is failed.
 @retval 0 The test is aborted by kind of system error.
 */
int FST_GME60x(struct gme_compass_data *data)
{
	int   pf_total;  //p/f flag for this subtest
	char    i2cData[16];
	int   hdata[3];
	int   asax;
	int   asay;
	int   asaz;
	int DRDY;
	//***********************************************
	//  Reset Test Result
	//***********************************************
	pf_total = 1;

	//***********************************************
	//  Step1
	//***********************************************

	// Reset device.
	if (AKECS_Reset(data,data->gpio_rstn) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	// Read values from WIA.
	i2cData[0] = GME60X_REG_WIA1;
	if (gme_rxdata(data->i2c,i2cData, 2) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	// TEST
	TEST_DATA(TLIMIT_NO_RST_WIA1_GME60x,   TLIMIT_TN_RST_WIA1_GME60x,   (int)i2cData[0],  TLIMIT_LO_RST_WIA1_GME60x,   TLIMIT_HI_RST_WIA1_GME60x,   &pf_total);
	TEST_DATA(TLIMIT_NO_RST_WIA2_GME60x,   TLIMIT_TN_RST_WIA2_GME60x,   (int)i2cData[1],  TLIMIT_LO_RST_WIA2_GME60x,   TLIMIT_HI_RST_WIA2_GME60x,   &pf_total);

	// Set to FUSE ROM access mode
	if (AKECS_SetMode(data,GME60X_MODE_FUSE_ACCESS) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	// Read values from ASAX to ASAZ
	i2cData[0] = GME60X_FUSE_ASAX;
	if (gme_rxdata(data->i2c,i2cData, 3) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}
	asax = (int)i2cData[0];
	asay = (int)i2cData[1];
	asaz = (int)i2cData[2];

	// TEST
	TEST_DATA(TLIMIT_NO_ASAX_GME60x, TLIMIT_TN_ASAX_GME60x, asax, TLIMIT_LO_ASAX_GME60x, TLIMIT_HI_ASAX_GME60x, &pf_total);
	TEST_DATA(TLIMIT_NO_ASAY_GME60x, TLIMIT_TN_ASAY_GME60x, asay, TLIMIT_LO_ASAY_GME60x, TLIMIT_HI_ASAY_GME60x, &pf_total);
	TEST_DATA(TLIMIT_NO_ASAZ_GME60x, TLIMIT_TN_ASAZ_GME60x, asaz, TLIMIT_LO_ASAZ_GME60x, TLIMIT_HI_ASAZ_GME60x, &pf_total);

	// Set to PowerDown mode
	if (AKECS_SetMode(data,GME60X_MODE_POWERDOWN) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}
	//***********************************************
	//  Step2
	//***********************************************

	// Set to SNG measurement pattern (Set CNTL register)
	if (AKECS_SetMode(data,GME60X_MODE_SNG_MEASURE) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	// Wait for DRDY pin changes to HIGH.
	for(DRDY=0;DRDY<10;DRDY++)
		udelay(AKM_MEASURE_TIME_US);
	// Get measurement data from GME60X
	// ST1 + (HXL + HXH) + (HYL + HYH) + (HZL + HZH) + TEMP + ST2
	// = 1 + (1 + 1) + (1 + 1) + (1 + 1) + 1 + 1 = 9yte
	//if (AKD_GetMagneticData(i2cData) != AKD_SUCCESS) {
	if (AKECS_GetData_Poll(data,i2cData,GME_SENSOR_DATA_SIZE) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	//hdata[0] = (int)((((uint)(i2cData[2]))<<8)+(uint)(i2cData[1]));
	//hdata[1] = (int)((((uint)(i2cData[4]))<<8)+(uint)(i2cData[3]));
	//hdata[2] = (int)((((uint)(i2cData[6]))<<8)+(uint)(i2cData[5]));
	
	hdata[0] = (s16)(i2cData[1] | (i2cData[2] << 8));
	hdata[1] = (s16)(i2cData[3] | (i2cData[4] << 8));
	hdata[2] = (s16)(i2cData[5] | (i2cData[6] << 8));
	
	// TEST
	i2cData[0] &= 0x7F;
	TEST_DATA(TLIMIT_NO_SNG_ST1_GME60x,  TLIMIT_TN_SNG_ST1_GME60x,  (int)i2cData[0], TLIMIT_LO_SNG_ST1_GME60x,  TLIMIT_HI_SNG_ST1_GME60x,  &pf_total);

	// TEST
	TEST_DATA(TLIMIT_NO_SNG_HX_GME60x,   TLIMIT_TN_SNG_HX_GME60x,   hdata[0],          TLIMIT_LO_SNG_HX_GME60x,   TLIMIT_HI_SNG_HX_GME60x,   &pf_total);
	TEST_DATA(TLIMIT_NO_SNG_HY_GME60x,   TLIMIT_TN_SNG_HY_GME60x,   hdata[1],          TLIMIT_LO_SNG_HY_GME60x,   TLIMIT_HI_SNG_HY_GME60x,   &pf_total);
	TEST_DATA(TLIMIT_NO_SNG_HZ_GME60x,   TLIMIT_TN_SNG_HZ_GME60x,   hdata[2],          TLIMIT_LO_SNG_HZ_GME60x,   TLIMIT_HI_SNG_HZ_GME60x,   &pf_total);
	TEST_DATA(TLIMIT_NO_SNG_ST2_GME60x,  TLIMIT_TN_SNG_ST2_GME60x,  (int)i2cData[8], TLIMIT_LO_SNG_ST2_GME60x,  TLIMIT_HI_SNG_ST2_GME60x,  &pf_total);

	// Set to Self-test mode (Set CNTL register)
	if (AKECS_SetMode(data,GME60X_MODE_SELF_TEST) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	// Wait for DRDY pin changes to HIGH.
	for(DRDY=0;DRDY<10;DRDY++)
		udelay(AKM_MEASURE_TIME_US);
	// Get measurement data from GME60X
	// ST1 + (HXL + HXH) + (HYL + HYH) + (HZL + HZH) + TEMP + ST2
	// = 1 + (1 + 1) + (1 + 1) + (1 + 1) + 1 + 1 = 9byte
	//if (AKD_GetMagneticData(i2cData) != AKD_SUCCESS) {
	if (AKECS_GetData_Poll(data,i2cData,GME_SENSOR_DATA_SIZE) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	// TEST
	i2cData[0] &= 0x7F;
	TEST_DATA(TLIMIT_NO_SLF_ST1_GME60x, TLIMIT_TN_SLF_ST1_GME60x, (int)i2cData[0], TLIMIT_LO_SLF_ST1_GME60x, TLIMIT_HI_SLF_ST1_GME60x, &pf_total);

	//hdata[0] = (int)((((uint)(i2cData[2]))<<8)+(uint)(i2cData[1]));
	//hdata[1] = (int)((((uint)(i2cData[4]))<<8)+(uint)(i2cData[3]));
	//hdata[2] = (int)((((uint)(i2cData[6]))<<8)+(uint)(i2cData[5]));

	hdata[0] = (s16)(i2cData[1] | (i2cData[2] << 8));
	hdata[1] = (s16)(i2cData[3] | (i2cData[4] << 8));
	hdata[2] = (s16)(i2cData[5] | (i2cData[6] << 8));

	// TEST
	TEST_DATA(
			  TLIMIT_NO_SLF_RVHX_GME60x,
			  TLIMIT_TN_SLF_RVHX_GME60x,
			  (hdata[0])*(asax/128 + 1),
			  TLIMIT_LO_SLF_RVHX_GME60x,
			  TLIMIT_HI_SLF_RVHX_GME60x,
			  &pf_total
			  );

	TEST_DATA(
			  TLIMIT_NO_SLF_RVHY_GME60x,
			  TLIMIT_TN_SLF_RVHY_GME60x,
			  (hdata[1])*(asay/128 + 1),
			  TLIMIT_LO_SLF_RVHY_GME60x,
			  TLIMIT_HI_SLF_RVHY_GME60x,
			  &pf_total
			  );

	TEST_DATA(
			  TLIMIT_NO_SLF_RVHZ_GME60x,
			  TLIMIT_TN_SLF_RVHZ_GME60x,
			  (hdata[2])*(asaz/128 + 1),
			  TLIMIT_LO_SLF_RVHZ_GME60x,
			  TLIMIT_HI_SLF_RVHZ_GME60x,
			  &pf_total
			  );

		TEST_DATA(
			TLIMIT_NO_SLF_ST2_GME60x,
			TLIMIT_TN_SLF_ST2_GME60x,
			(int)i2cData[8],
			TLIMIT_LO_SLF_ST2_GME60x,
			TLIMIT_HI_SLF_ST2_GME60x,
			&pf_total
			);
	// Reset device.
	if (AKECS_Reset(data,data->gpio_rstn) < 0) {
		GMEDBG("%s:%d Error.\n", __FUNCTION__, __LINE__);
		return 0;
	}

	return pf_total;
}


int FctShipmntTestProcess_Body(struct gme_compass_data *data)
{
	int pf_total = 1;

	//***********************************************
	//    Reset Test Result
	//***********************************************
	TEST_DATA(NULL, "START", 0, 0, 0, &pf_total);

	//***********************************************
	//    Step 1 to 2
	//***********************************************
	pf_total = FST_GME60x(data);
	//***********************************************
	//    Judge Test Result
	//***********************************************
	TEST_DATA(NULL, "END", 0, 0, 0, &pf_total);

	return pf_total;
}

static ssize_t store_shipment_test(
	struct device *dev, struct device_attribute *attr,char const *buf, size_t count)
{
	
	return count;            
}

static ssize_t show_shipment_test(struct device *dev, struct device_attribute *attr, char *buf)
{
	char result[10];
	int res = 0;
	struct gme_compass_data *data=dev_get_drvdata(dev);
	res = FctShipmntTestProcess_Body(data);
	if(1 == res)
	{
	   GMEDBG("shipment_test pass\n");
	   strcpy(result,"y");
	}
	else if(-1 == res)
	{
	  GMEDBG("shipment_test fail\n");
	   strcpy(result,"n");
	}
	else
	{
	  GMEDBG("shipment_test NaN\n");
	  strcpy(result,"NaN");
	}
	
	return sprintf(buf, "%s\n", result);        
}

/***** Fusion ***/
static ssize_t gme_delay_fusion_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	return gme_compass_sysfs_delay_show(
		dev_get_drvdata(dev), buf, FUSION_DATA_FLAG);
}
static ssize_t gme_delay_fusion_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	return gme_compass_sysfs_delay_store(
		dev_get_drvdata(dev), buf, count, FUSION_DATA_FLAG);
}

/***** accel (binary) ***/
static ssize_t gme_bin_accel_write(
	struct file *file,
	struct kobject *kobj,
	struct bin_attribute *attr,
		char *buf,
		loff_t pos,
		size_t size)
{
	struct device *dev = container_of(kobj, struct device, kobj);
	struct gme_compass_data *data = dev_get_drvdata(dev);
	int16_t *accel_data;

	if (size == 0)
		return 0;

	accel_data = (int16_t *)buf;

	mutex_lock(&data->accel_mutex);
	data->accel_data[0] = accel_data[0];
	data->accel_data[1] = accel_data[1];
	data->accel_data[2] = accel_data[2];
	mutex_unlock(&data->accel_mutex);
/*
	dev_dbg(&data->i2c->dev, "kernel accel:%d,%d,%d\n",
			accel_data[0], accel_data[1], accel_data[2]);
*/
	return size;
}


#if GME_DEBUG_IF
static ssize_t gme_sysfs_mode_store(
	struct device *dev, struct device_attribute *attr,
	char const *buf, size_t count)
{
	struct gme_compass_data *data = dev_get_drvdata(dev);
	long mode = 0;

	if (NULL == buf)
		return -EINVAL;

	if (0 == count)
		return 0;

	if (strict_strtol(buf, GME_BASE_NUM, &mode))
		return -EINVAL;

	if (AKECS_SetMode(data, (uint8_t)mode) < 0)
		return -EINVAL;

	return 1;
}

static ssize_t gme_buf_print(
	char *buf, uint8_t *data, size_t num)
{
	int sz, i;
	char *cur;
	size_t cur_len;

	cur = buf;
	cur_len = PAGE_SIZE;
	sz = snprintf(cur, cur_len, "(HEX):");
	if (sz < 0)
		return sz;
	cur += sz;
	cur_len -= sz;
	for (i = 0; i < num; i++) {
		sz = snprintf(cur, cur_len, "%02X,", *data);
		if (sz < 0)
			return sz;
		cur += sz;
		cur_len -= sz;
		data++;
	}
	sz = snprintf(cur, cur_len, "\n");
	if (sz < 0)
		return sz;
	cur += sz;

	return (ssize_t)(cur - buf);
}

static ssize_t gme_sysfs_bdata_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	struct gme_compass_data *data = dev_get_drvdata(dev);
	uint8_t rbuf[GME_SENSOR_DATA_SIZE];

	mutex_lock(&data->sensor_mutex);
	memcpy(&rbuf, data->sense_data, sizeof(rbuf));
	mutex_unlock(&data->sensor_mutex);

	return gme_buf_print(buf, rbuf, GME_SENSOR_DATA_SIZE);
}

static ssize_t gme_sysfs_asa_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	struct gme_compass_data *data = dev_get_drvdata(dev);
	int err;
	uint8_t asa[3];

	err = AKECS_SetMode(data, GME_MODE_FUSE_ACCESS);
	if (err < 0)
		return err;

	asa[0] = GME_FUSE_1ST_ADDR;
	err = gme_rxdata(data->i2c, asa, 3);
	if (err < 0)
		return err;

	err = AKECS_SetMode(data, GME_MODE_POWERDOWN);
	if (err < 0)
		return err;

	return gme_buf_print(buf, asa, 3);
}

static ssize_t gme_sysfs_regs_show(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	/* The total number of registers depends on the device. */
	struct gme_compass_data *data = dev_get_drvdata(dev);
	int err;
	uint8_t regs[GME_REGS_SIZE];

	/* This function does not lock mutex obj */
	regs[0] = GME_REGS_1ST_ADDR;
	err = gme_rxdata(data->i2c, regs, GME_REGS_SIZE);
	if (err < 0)
		return err;

	return gme_buf_print(buf, regs, GME_REGS_SIZE);
}
#endif

static struct device_attribute gme_compass_attributes[] = {
	__ATTR(enable_acc, 0660, gme_enable_acc_show, gme_enable_acc_store),
	__ATTR(enable_mag, 0660, gme_enable_mag_show, gme_enable_mag_store),
	__ATTR(enable_fusion, 0660, gme_enable_fusion_show,gme_enable_fusion_store),
	__ATTR(delay_acc,  0660, gme_delay_acc_show,  gme_delay_acc_store),
	__ATTR(delay_mag,  0660, gme_delay_mag_show,  gme_delay_mag_store),
	__ATTR(delay_fusion, 0660, gme_delay_fusion_show,gme_delay_fusion_store),
	__ATTR(shipmenttest, 0660, show_shipment_test, store_shipment_test),
#if GME_DEBUG_IF
	__ATTR(mode,  0220, NULL, gme_sysfs_mode_store),
	__ATTR(bdata, 0440, gme_sysfs_bdata_show, NULL),
	__ATTR(asa,   0440, gme_sysfs_asa_show, NULL),
	__ATTR(regs,  0440, gme_sysfs_regs_show, NULL),
#endif
	__ATTR_NULL,
};

#define __BIN_ATTR(name_, mode_, size_, private_, read_, write_) \
	{ \
		.attr    = { .name = __stringify(name_), .mode = mode_ }, \
		.size    = size_, \
		.private = private_, \
		.read    = read_, \
		.write   = write_, \
	}

#define __BIN_ATTR_NULL \
	{ \
		.attr   = { .name = NULL }, \
	}

static struct bin_attribute gme_compass_bin_attributes[] = {
	__BIN_ATTR(accel, 0220, 6, NULL,
				NULL, gme_bin_accel_write),
	__BIN_ATTR_NULL
};

static char const *const device_link_name = "i2c";
static dev_t const gme_compass_device_dev_t = MKDEV(MISC_MAJOR, 244);

static int create_sysfs_interfaces(struct gme_compass_data *data)
{
	int err;

	if (NULL == data)
		return -EINVAL;

	err = 0;

	data->compass = class_create(THIS_MODULE, GME_SYSCLS_NAME);
	if (IS_ERR(data->compass)) {
		err = PTR_ERR(data->compass);
		goto exit_class_create_failed;
	}

	data->class_dev = device_create(
						data->compass,
						NULL,
						gme_compass_device_dev_t,
						data,
						GME_SYSDEV_NAME);
	if (IS_ERR(data->class_dev)) {
		err = PTR_ERR(data->class_dev);
		goto exit_class_device_create_failed;
	}

	err = sysfs_create_link(
			&data->class_dev->kobj,
			&data->i2c->dev.kobj,
			device_link_name);
	if (0 > err)
		goto exit_sysfs_create_link_failed;

	err = create_device_attributes(
			data->class_dev,
			gme_compass_attributes);
	if (0 > err)
		goto exit_device_attributes_create_failed;

	err = create_device_binary_attributes(
			&data->class_dev->kobj,
			gme_compass_bin_attributes);
	if (0 > err)
		goto exit_device_binary_attributes_create_failed;

	return err;

exit_device_binary_attributes_create_failed:
	remove_device_attributes(data->class_dev, gme_compass_attributes);
exit_device_attributes_create_failed:
	sysfs_remove_link(&data->class_dev->kobj, device_link_name);
exit_sysfs_create_link_failed:
	device_destroy(data->compass, gme_compass_device_dev_t);
exit_class_device_create_failed:
	data->class_dev = NULL;
	class_destroy(data->compass);
exit_class_create_failed:
	data->compass = NULL;
	return err;
}

static void remove_sysfs_interfaces(struct gme_compass_data *data)
{
	if (NULL == data)
		return;

	if (NULL != data->class_dev) {
		remove_device_binary_attributes(
			&data->class_dev->kobj,
			gme_compass_bin_attributes);
		remove_device_attributes(
			data->class_dev,
			gme_compass_attributes);
		sysfs_remove_link(
			&data->class_dev->kobj,
			device_link_name);
		data->class_dev = NULL;
	}
	if (NULL != data->compass) {
		device_destroy(
			data->compass,
			gme_compass_device_dev_t);
		class_destroy(data->compass);
		data->compass = NULL;
	}
}

/***** globalmems input device functions ***********************************/
static int gme_compass_input_init(
	struct input_dev **input)
{
	int err = 0;

	/* Declare input device */
	*input = input_allocate_device();
	if (!*input)
		return -ENOMEM;

	/* Setup input device */
	set_bit(EV_ABS, (*input)->evbit);
	/* Accelerometer (720 x 16G)*/
	input_set_abs_params(*input, ABS_X,
			-11520, 11520, 0, 0);
	input_set_abs_params(*input, ABS_Y,
			-11520, 11520, 0, 0);
	input_set_abs_params(*input, ABS_Z,
			-11520, 11520, 0, 0);
	input_set_abs_params(*input, ABS_RX,
			0, 3, 0, 0);
	/* Magnetic field (limited to 16bit) */
	input_set_abs_params(*input, ABS_RY,
			-32768, 32767, 0, 0);
	input_set_abs_params(*input, ABS_RZ,
			-32768, 32767, 0, 0);
	input_set_abs_params(*input, ABS_THROTTLE,
			-32768, 32767, 0, 0);
	input_set_abs_params(*input, ABS_RUDDER,
			0, 3, 0, 0);

	/* Orientation (degree in Q6 format) */
	/*  yaw[0,360) pitch[-180,180) roll[-90,90) */
	input_set_abs_params(*input, ABS_HAT0Y,
			0, 23040, 0, 0);
	input_set_abs_params(*input, ABS_HAT1X,
			-11520, 11520, 0, 0);
	input_set_abs_params(*input, ABS_HAT1Y,
			-5760, 5760, 0, 0);
	/* Rotation Vector [-1,+1] in Q14 format */
	input_set_abs_params(*input, ABS_TILT_X,
			-16384, 16384, 0, 0);
	input_set_abs_params(*input, ABS_TILT_Y,
			-16384, 16384, 0, 0);
	input_set_abs_params(*input, ABS_TOOL_WIDTH,
			-16384, 16384, 0, 0);
	input_set_abs_params(*input, ABS_VOLUME,
			-16384, 16384, 0, 0);
			
 	/* soft gyro*/
	input_set_abs_params(*input, ABS_HAT2X,
			-32767, 32767, 0, 0);
	input_set_abs_params(*input, ABS_HAT2Y,
			-32767, 32767, 0, 0);
	input_set_abs_params(*input, ABS_HAT3X,
			-32767, 32767, 0, 0);

	/* Set name */
	(*input)->name = GME_INPUT_DEVICE_NAME;

	/* Register */
	err = input_register_device(*input);
	if (err) {
		input_free_device(*input);
		return err;
	}

	return err;
}

/***** globalmems functions ************************************************/
static irqreturn_t gme_compass_irq(int irq, void *handle)
{
	struct gme_compass_data *data = handle;
	uint8_t buffer[GME_SENSOR_DATA_SIZE];
	int err;

	memset(buffer, 0, sizeof(buffer));

	/***** lock *****/
	mutex_lock(&data->sensor_mutex);

	/* Read whole data */
	buffer[0] = GME_REG_STATUS;
	err = gme_rxdata(data->i2c, buffer, GME_SENSOR_DATA_SIZE);
	if (err < 0) {
		dev_err(&data->i2c->dev, "IRQ I2C error.");
		data->is_busy = 0;
		mutex_unlock(&data->sensor_mutex);
		/***** unlock *****/

		return IRQ_HANDLED;
	}
	/* Check ST bit */
	if (!(GME_DRDY_IS_HIGH(buffer[0])))
		goto work_func_none;

	memcpy(data->sense_data, buffer, GME_SENSOR_DATA_SIZE);
	data->is_busy = 0;

	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

	atomic_set(&data->drdy, 1);
	wake_up(&data->drdy_wq);

	dev_vdbg(&data->i2c->dev, "IRQ handled.");
	return IRQ_HANDLED;

work_func_none:
	mutex_unlock(&data->sensor_mutex);
	/***** unlock *****/

	dev_vdbg(&data->i2c->dev, "IRQ not handled.");
	return IRQ_NONE;
}

static int gme_compass_suspend(struct device *dev)
{
	struct gme_compass_data *data = dev_get_drvdata(dev);
	dev_dbg(&data->i2c->dev, "suspended\n");

	return 0;
}

static int gme_compass_resume(struct device *dev)
{
	struct gme_compass_data *data = dev_get_drvdata(dev);
	dev_dbg(&data->i2c->dev, "resumed\n");

	return 0;
}

static int gme60x_i2c_check_device(
	struct i2c_client *client)
{
	/* GME60x specific function */
	struct gme_compass_data *data = i2c_get_clientdata(client);
	int err;

	data->sense_info[0] = GME60X_REG_WIA1;
	err = gme_rxdata(client, data->sense_info, GME_SENSOR_INFO_SIZE);
	if (err < 0)
		return err;

	/* Set FUSE access mode */
	err = AKECS_SetMode(data, GME60X_MODE_FUSE_ACCESS);
	if (err < 0)
		return err;

	data->sense_conf[0] = GME60X_FUSE_ASAX;
	err = gme_rxdata(client, data->sense_conf, GME_SENSOR_CONF_SIZE);
	if (err < 0)
		return err;

	err = AKECS_SetMode(data, GME60X_MODE_POWERDOWN);
	if (err < 0)
		return err;

	/* Check read data */
	if ((data->sense_info[0] != GME60X_WIA1_VALUE) ||
			(data->sense_info[1] != GME60X_WIA2_VALUE)){
		dev_err(&client->dev,
			"%s: The device is not GLOBALMEMS Compass.", __func__);
		return -ENXIO;
	}

	return err;
}

int gme_compass_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	//struct gme60x_platform_data *pdata;
	int err = 0;
	int i;

	dev_dbg(&client->dev, "start probing.");

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		dev_err(&client->dev,
				"%s: check_functionality failed.", __func__);
		err = -ENODEV;
		goto exit0;
	}

	/* Allocate memory for driver data */
	s_gme = kzalloc(sizeof(struct gme_compass_data), GFP_KERNEL);
	if (!s_gme) {
		dev_err(&client->dev,
				"%s: memory allocation failed.", __func__);
		err = -ENOMEM;
		goto exit1;
	}

	/**** initialize variables in gme_compass_data *****/
	init_waitqueue_head(&s_gme->drdy_wq);
	init_waitqueue_head(&s_gme->open_wq);

	mutex_init(&s_gme->sensor_mutex);
	mutex_init(&s_gme->accel_mutex);
	mutex_init(&s_gme->val_mutex);

	atomic_set(&s_gme->active, 0);
	atomic_set(&s_gme->drdy, 0);

	s_gme->is_busy = 0;
	s_gme->enable_flag = 0;

	/* Set to 1G in Android coordination, AKSC format */
	s_gme->accel_data[0] = 0;
	s_gme->accel_data[1] = 0;
	s_gme->accel_data[2] = 720;

	for (i = 0; i < GME_NUM_SENSORS; i++)
		s_gme->delay[i] = -1;

	/***** Set platform information *****/
/*	pdata = client->dev.platform_data;
	if (pdata) {
		// Platform data is available. copy its value to local.
		s_gme->layout = pdata->layout;
		s_gme->gpio_rstn = pdata->gpio_RSTN;
	} else {
		// Platform data is not available.
		   Layout and information should be set by each application.
		dev_dbg(&client->dev, "%s: No platform data.", __func__);
		s_gme->layout = 0;
		s_gme->gpio_rstn = 0;
	}
*/
	s_gme->layout =4;
	s_gme->gpio_rstn = 6;
	//Add for rst pin test fred+
	if(!(gpio_request(s_gme->gpio_rstn, "ECOMPASS_RST")))
	{
	  GMEDBG("[fred_test]ECOMPASS_RST repuest success");
	  if(!(gpio_direction_output(s_gme->gpio_rstn, 1)))
	  { 
	    GMEDBG("[fred_test]ECOMPASS_RST dir pass");
	    gpio_set_value(s_gme->gpio_rstn, 1);
	  }
	  else
	    GMEDBG("[fred_test]ECOMPASS_RST dir fail");
	}
	else
	  GMEDBG("[fred_test]ECOMPASS_RST repuest fail");
	//Add for rst pin test fred-
	dev_dbg(&client->dev, "%s,%d: Layout=%d", __func__,__LINE__,s_gme->layout);
	/***** I2C initialization *****/
	s_gme->i2c = client;
	/* set client data */
	i2c_set_clientdata(client, s_gme);
	/* check connection */
	err = gme60x_i2c_check_device(client);
	if (err < 0)
		goto exit2;

	/***** input *****/
	err = gme_compass_input_init(&s_gme->input);
	if (err) {
		dev_err(&client->dev,
			"%s: input_dev register failed", __func__);
		goto exit3;
	}
	input_set_drvdata(s_gme->input, s_gme);

	/***** IRQ setup *****/
	s_gme->irq = client->irq;

	dev_dbg(&client->dev, "%s: IRQ is #%d.",
			__func__, s_gme->irq);

	if (s_gme->irq) {
		err = request_threaded_irq(
				s_gme->irq,
				NULL,
				gme_compass_irq,
				IRQF_TRIGGER_HIGH|IRQF_ONESHOT,
				dev_name(&client->dev),
				s_gme);
		if (err < 0) {
			dev_err(&client->dev,
				"%s: request irq failed.", __func__);
			goto exit4;
		}
	}

	/***** misc *****/
	err = misc_register(&gme_compass_dev);
	if (err) {
		dev_err(&client->dev,
			"%s: GLOBALMEMS_compass_dev register failed", __func__);
		goto exit5;
	}

	/***** sysfs *****/
	err = create_sysfs_interfaces(s_gme);
	if (0 > err) {
		dev_err(&client->dev,
			"%s: create sysfs failed.", __func__);
		goto exit6;
	}

	dev_info(&client->dev, "successfully probed.");
	return 0;

exit6:
	misc_deregister(&gme_compass_dev);
exit5:
	if (s_gme->irq)
		free_irq(s_gme->irq, s_gme);
exit4:
	input_unregister_device(s_gme->input);
exit3:
exit2:
	kfree(s_gme);
exit1:
exit0:
	return err;
}

static int gme_compass_remove(struct i2c_client *client)
{
	struct gme_compass_data *data = i2c_get_clientdata(client);

	remove_sysfs_interfaces(data);
	if (misc_deregister(&gme_compass_dev) < 0)
		dev_err(&client->dev, "misc deregister failed.");
	if (data->irq)
		free_irq(data->irq, data);
	input_unregister_device(data->input);
	kfree(data);
	dev_info(&client->dev, "successfully removed.");
	return 0;
}

static const struct i2c_device_id gme_compass_id[] = {
	{GME_I2C_NAME, 0 },
	{ }
};
//Add for QCOM DTS   fred 20150831+++
#if QCOM
static struct of_device_id match_table[] = {
	{ .compatible = "magnetic,gmc303",},
	{ },
};
#endif
//Add for QCOM DTS   fred 20150831---

static const struct dev_pm_ops gme_compass_pm_ops = {
	.suspend	= gme_compass_suspend,
	.resume		= gme_compass_resume,
};

static struct i2c_driver gme_compass_driver = {
	.probe		= gme_compass_probe,
	.remove		= gme_compass_remove,
	.id_table	= gme_compass_id,
	.driver = {
		.name	= GME_I2C_NAME,
		.pm		= &gme_compass_pm_ops,
//Add for QCOM DTS   fred 20150831+++
#if QCOM
		.of_match_table = match_table,
#endif
//Add for QCOM DTS   fred 20150831---
	},
};

static int __init gme_compass_init(void)
{
	pr_info("GLOBALMEMS compass driver: initialize.");
	return i2c_add_driver(&gme_compass_driver);
}

static void __exit gme_compass_exit(void)
{
	pr_info("GLOBALMEMS compass driver: release.");
	i2c_del_driver(&gme_compass_driver);
}

module_init(gme_compass_init);
module_exit(gme_compass_exit);

MODULE_AUTHOR("viral wang <viral_wang@htc.com>");
MODULE_DESCRIPTION("GLOBALMEMS compass driver");
MODULE_LICENSE("GPL");
MODULE_VERSION("0.4");

